#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SH110X.h>

// ---------- Display ----------
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
Adafruit_SH1106G display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ---------- Buttons ----------
const int BTN_LEFT  = D5; // GPIO14
const int BTN_RIGHT = D6; // GPIO12

// ---------- YOUR BITMAP from imag2.cpp ----------
// 👉 IMPORTANT: Set these to the REAL size of your bitmap
#define IMG_WIDTH   128   // <-- change to your width
#define IMG_HEIGHT  64   // <-- change to your height

/*
 * 👉 Paste your array from imag2.cpp here.
 * The safest form on ESP8266 is: const uint8_t PROGMEM myBitmapUntitled_1[] = { ... };
 * If your converter gave: "const unsigned char myBitmapUntitled_1[] PROGMEM = { ... };"
 * that also works.
 */
// 'Untitled-1', 128x64px
// 'Untitled-1', 128x64px
const unsigned char myBitmapUntitled_1 [] PROGMEM = {
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x7f, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x3f, 0xff, 0xe1, 0xff, 0xff, 0x8f, 0xff, 0xfc, 0x7f, 0xff, 0xe3, 0xff, 0xff, 0x8f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0xff, 0xff, 0xe3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 
	0x7f, 0xff, 0xf1, 0xff, 0xff, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x7f, 0xff, 0xf3, 0xff, 0xff, 0x9f, 0xff, 0xfc, 0x00, 0x00, 0x03, 0xff, 0xff, 0x80, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x1f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 
	0x1f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xfe, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};



// Array of all bitmaps for convenience. (Total bytes used to store images in PROGMEM = 1040)
const int myBitmapallArray_LEN = 1;
const unsigned char* myBitmapallArray[1] = {
	myBitmapUntitled_1
};

// Small utility to draw bitmap centered
void drawBitmapCentered(const uint8_t *bmp, int bw, int bh) {
  int x = (SCREEN_WIDTH  - bw) / 2;
  int y = (SCREEN_HEIGHT - bh) / 2;
  display.drawBitmap(x, y, bmp, bw, bh, SH110X_WHITE);
}

// ---------- Game params (slow + smooth) ----------
int   paddleW = 24;
int   paddleH = 3;
int   paddleY = SCREEN_HEIGHT - 6;
float paddleX = (SCREEN_WIDTH - 24) / 2.0;
float paddleSpeed = 1.8;

float ballX = SCREEN_WIDTH / 2.0;
float ballY = SCREEN_HEIGHT / 2.0;
float ballVX = 0.9;
float ballVY = -1.2;
int   ballR  = 2;

const int BR_ROWS   = 4;
const int BR_COLS   = 10;
const int BR_W      = 10;
const int BR_H      = 5;
const int BR_HSPACE = 2;
const int BR_VSPACE = 2;

const int BR_FIELD_W = BR_COLS * BR_W + (BR_COLS - 1) * BR_HSPACE;
const int BR_START_X = (SCREEN_WIDTH - BR_FIELD_W) / 2;
const int BR_START_Y = 10;

bool bricks[BR_ROWS][BR_COLS];
int  bricksLeft = BR_ROWS * BR_COLS;

int   lives = 3;
long  lastFrame = 0;
const int FRAME_MS = 16;
long  lastInputDebounce = 0;
const int INPUT_DEBOUNCE_MS = 5;
long  score = 0;
int   level = 1;

// ---------- Helpers ----------
bool rectCircleCollide(int rx,int ry,int rw,int rh,float cx,float cy,int cr) {
  float tx=cx, ty=cy;
  if (cx < rx)         tx = rx;
  else if (cx > rx+rw) tx = rx+rw;
  if (cy < ry)         ty = ry;
  else if (cy > ry+rh) ty = ry+rh;
  float dx=cx-tx, dy=cy-ty;
  return (dx*dx + dy*dy) <= (cr*cr);
}

void drawBricks() {
  for (int r=0; r<BR_ROWS; r++) {
    for (int c=0; c<BR_COLS; c++) {
      if (!bricks[r][c]) continue;
      int x = BR_START_X + c*(BR_W + BR_HSPACE);
      int y = BR_START_Y + r*(BR_H + BR_VSPACE);
      display.fillRect(x, y, BR_W, BR_H, SH110X_WHITE);
      display.drawRect(x, y, BR_W, BR_H, SH110X_BLACK);
    }
  }
}

void buildBricks() {
  for (int r=0; r<BR_ROWS; r++)
    for (int c=0; c<BR_COLS; c++)
      bricks[r][c] = true;
  bricksLeft = BR_ROWS * BR_COLS;
}

void resetBall(bool fromCenter = true) {
  if (fromCenter) { ballX = SCREEN_WIDTH/2.0; ballY = SCREEN_HEIGHT/2.0; }
  else            { ballX = paddleX + paddleW/2.0; ballY = paddleY - ballR - 1; }
  ballVX = (random(0,2)==0) ? 0.9 : -0.9;
  ballVY = -1.2;
}

void resetLevel(bool fullReset) {
  if (fullReset) {
    level = 1; score = 0; lives = 3;
    paddleW = 24; paddleSpeed = 1.8;
    ballVX = 0.9; ballVY = -1.2;
  } else {
    level++;
    ballVX *= 1.05; ballVY *= 1.05;            // slow ramp
    if (paddleW > 16) paddleW -= 2;            // a little harder
  }
  paddleX = (SCREEN_WIDTH - paddleW) / 2.0;
  buildBricks();
  resetBall(true);
}

void handleBrickCollision() {
  float prevX = ballX - ballVX;
  float prevY = ballY - ballVY;
  for (int r=0; r<BR_ROWS; r++) {
    for (int c=0; c<BR_COLS; c++) {
      if (!bricks[r][c]) continue;
      int bx = BR_START_X + c*(BR_W+BR_HSPACE);
      int by = BR_START_Y + r*(BR_H+BR_VSPACE);
      if (rectCircleCollide(bx, by, BR_W, BR_H, ballX, ballY, ballR)) {
        bool prevInsideX = (prevX >= bx && prevX <= bx + BR_W);
        bool prevInsideY = (prevY >= by && prevY <= by + BR_H);
        if (prevInsideX && !prevInsideY) ballVY = -ballVY;
        else if (!prevInsideX && prevInsideY) ballVX = -ballVX;
        else ballVY = -ballVY;
        bricks[r][c] = false;
        bricksLeft--;
        score += 10;
        return;
      }
    }
  }
}

void waitForAnyButton() {
  display.clearDisplay();
  display.setTextSize(1);                      // NORMAL FONT
  display.setTextColor(SH110X_WHITE);
  // centered-ish text
  display.setCursor(17, 26);
  display.println(F("Press any button"));
  display.setCursor(40, 38);
  display.println(F("to start"));
  display.display();
  while (digitalRead(BTN_LEFT)==HIGH && digitalRead(BTN_RIGHT)==HIGH) delay(5);
}

// ---------- BOOT: Image with title underneath ----------
void bootSequence() {
  display.clearDisplay();

  // Step 1: show image
  drawBitmapCentered(myBitmapUntitled_1, IMG_WIDTH, IMG_HEIGHT);

  // Step 2: title text below image (x = 0, y = 64)
  display.setTextSize(1);                    // normal font
  display.setTextColor(SH110X_WHITE);
  // center text horizontally
  const char* title = "Break Wall By JP";
  int16_t x = (SCREEN_WIDTH - strlen(title) * 6) / 2;
  if (x < 0) x = 0;
  display.setCursor(x, 48);                  // OLED height = 64 → baseline ≈ 56
  display.print(title);

  display.display();
  delay(1200);

  // Step 3: press-to-start prompt
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SH110X_WHITE);
  display.setCursor(17, 26);
  display.println(F("Press any button"));
  display.setCursor(40, 38);
  display.println(F("to start"));
  display.display();

  // wait until a button is pressed
  while (digitalRead(BTN_LEFT)==HIGH && digitalRead(BTN_RIGHT)==HIGH) delay(5);
}


void setup() {
  pinMode(BTN_LEFT,  INPUT_PULLUP);
  pinMode(BTN_RIGHT, INPUT_PULLUP);

  Wire.begin();
  display.begin(0x3C, true);         // try 0x3D if blank
  display.setContrast(160);
  randomSeed(ESP.getChipId());

  bootSequence();
  resetLevel(true);
  lastFrame = millis();
}

void loop() {
  long now = millis();
  if (now - lastFrame < FRAME_MS) return;
  lastFrame = now;

  // Input
  bool leftPressed  = (digitalRead(BTN_LEFT)  == LOW);
  bool rightPressed = (digitalRead(BTN_RIGHT) == LOW);
  if (now - lastInputDebounce > INPUT_DEBOUNCE_MS) {
    if (leftPressed && !rightPressed)      paddleX -= paddleSpeed;
    else if (rightPressed && !leftPressed) paddleX += paddleSpeed;
    lastInputDebounce = now;
  }
  if (paddleX < 0) paddleX = 0;
  if (paddleX + paddleW > SCREEN_WIDTH) paddleX = SCREEN_WIDTH - paddleW;

  // Physics
  ballX += ballVX;
  ballY += ballVY;

  if (ballX - ballR < 0) { ballX = ballR; ballVX = -ballVX; }
  if (ballX + ballR > SCREEN_WIDTH) { ballX = SCREEN_WIDTH - ballR; ballVX = -ballVX; }
  if (ballY - ballR < 0) { ballY = ballR; ballVY = -ballVY; }

  // Bottom -> lose life
  if (ballY - ballR > SCREEN_HEIGHT) {
    lives--;
    if (lives <= 0) {
      display.clearDisplay();
      display.setTextSize(1);                // NORMAL FONT
      display.setCursor(34, 24);
      display.println(F("GAME OVER"));
      display.setCursor(34, 38);
      display.print(F("Score: "));
      display.print(score);
      display.display();
      delay(1400);

      bootSequence();
      resetLevel(true);
    } else {
      ballX = paddleX + paddleW/2.0;
      ballY = paddleY - ballR - 1;
      ballVX = (random(0,2)==0)? 0.9 : -0.9;
      ballVY = -1.2;
    }
  }

  // Paddle bounce
  if (rectCircleCollide((int)paddleX, paddleY, paddleW, paddleH, ballX, ballY, ballR)) {
    ballY = paddleY - ballR - 0.1;
    ballVY = -fabs(ballVY);
    float hit = (ballX - (paddleX + paddleW/2.0)) / (paddleW/2.0);
    ballVX += hit * 0.6;
    if (ballVX > 2.2)  ballVX = 2.2;
    if (ballVX < -2.2) ballVX = -2.2;
  }

  // Bricks + level
  if (bricksLeft > 0) handleBrickCollision();
  if (bricksLeft == 0) {
    resetLevel(false);
    display.clearDisplay();
    display.setTextSize(1);
    display.setCursor(44, 28);
    display.print(F("Level "));
    display.print(level);
    display.display();
    delay(500);
  }

  // Draw HUD + game
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SH110X_WHITE);
  display.setCursor(0, 0);
  display.print(F("L:")); display.print(lives);
  display.setCursor(28, 0);
  display.print(F("Lv:")); display.print(level);
  display.setCursor(70, 0);
  display.print(F("Score:")); display.print(score);

  drawBricks();
  display.fillRect((int)paddleX, paddleY, paddleW, paddleH, SH110X_WHITE);
  display.fillCircle((int)round(ballX), (int)round(ballY), ballR, SH110X_WHITE);

  display.display();
}
