#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#include <Fonts/FreeSansOblique12pt7b.h>  // Italic for "Boring"
#include <Fonts/FreeSansBold12pt7b.h>     // Bold for "Ball"
#include <math.h>
#include <Fonts/FreeSansBoldOblique12pt7b.h>

// ===== OLED =====
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET   -1
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// ===== Pins =====
const int BUTTON_PIN = D5; // one side to D5, other to GND

// ===== Physics =====
const float GRAVITY   = 0.35f;  // px/frame^2
const float DAMP_WALL = 0.90f;  // side walls
const float DAMP_TOP  = 0.92f;  // top bounce (slight)
const uint8_t GROUND_Y = SCREEN_HEIGHT - 1; // 63

// ===== Bitmaps from you =====
const int BALL_W = 16;
const int BALL_H = 16;
const unsigned char epd_bitmap_8888 [] PROGMEM = {
  0x00, 0x00, 0x07, 0xe0, 0x0f, 0xf0, 0x1f, 0xf8, 0x3f, 0xfc, 0x7f, 0xfe, 0x7f, 0xfe, 0x7f, 0xfe, 
	0x7f, 0xfe, 0x7f, 0xfe, 0x7f, 0xfe, 0x3f, 0xfc, 0x1f, 0xf8, 0x0f, 0xf0, 0x07, 0xe0, 0x00, 0x00
};

const unsigned char epd_bitmap_Untitled_2 [] PROGMEM = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x07, 0xff, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0xff, 0x80, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0xff, 0x80, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1d, 0xff, 0xc0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0xff, 0xc0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x67, 0xff, 0x70, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x63, 0xfc, 0x30, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x81, 0xfc, 0x10, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc7, 0xaf, 0x30, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x23, 0xf0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x61, 0xe0, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x00, 
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00
};

// ===== Game State =====
float x, y, vx = 0, vy = 0;
bool  onGround = true;

// Debounce
unsigned long lastPressMs = 0;
const unsigned long DEBOUNCE_MS = 120;
bool buttonJustPressed() {
  static bool last = false;
  bool curr = (digitalRead(BUTTON_PIN) == LOW);
  bool edge = (curr && !last && (millis() - lastPressMs) > DEBOUNCE_MS);
  if (edge) lastPressMs = millis();
  last = curr;
  return edge;
}

// ------- Title (Welcome) -------
void drawWelcome() {
  display.clearDisplay();
  display.setFont(&FreeSansBoldOblique12pt7b);
  display.setTextColor(WHITE);

  // Line 1: "Boring" (center)
  const char* l1 = "Boring";
  int16_t x1, y1; uint16_t w1, h1;
  display.getTextBounds(l1, 0, 0, &x1, &y1, &w1, &h1);
  int16_t cx1 = (SCREEN_WIDTH - w1) / 2;
  int16_t cy1 = 22;                 // baseline y (tweak if needed)
  display.setCursor(cx1, cy1);
  display.print(l1);

  // Line 2: "Ball" (center, next line)
  const char* l2 = "Ball";
  int16_t x2, y2; uint16_t w2, h2;
  display.getTextBounds(l2, 0, 0, &x2, &y2, &w2, &h2);
  int16_t cx2 = (SCREEN_WIDTH - w2) / 2;
  int16_t gap = 6;                  // line gap
  int16_t cy2 = cy1 + h1 + gap;     // next line baseline
  display.setCursor(cx2, cy2);
  display.print(l2);

  display.display();
}
void drawWelcome2() {
  display.clearDisplay();
  // First line: "Boring" italic
  display.setFont(&FreeSansOblique12pt7b);
  const char* line1 = "Boring";
  int16_t x1,y1; uint16_t w1,h1;
  display.getTextBounds(line1, 0, 0, &x1, &y1, &w1, &h1);
  int16_t cx1 = (SCREEN_WIDTH - w1)/2;
  int16_t cy1 = 20;             // baseline y
  display.setCursor(cx1, cy1);
  display.setTextColor(WHITE);
  display.print(line1);

  // Second line: "Ball" bold
  display.setFont(&FreeSansBold12pt7b);
  const char* line2 = "Ball";
  int16_t x2,y2; uint16_t w2,h2;
  display.getTextBounds(line2, 0, 0, &x2, &y2, &w2, &h2);
  int16_t cx2 = (SCREEN_WIDTH - w2)/2;
  int16_t cy2 = 42;             // next line baseline
  display.setCursor(cx2, cy2);
  display.print(line2);

  display.display();
}

// ------- Rendering -------
void drawBackground() {
  // If you want exact bg write (both 1 and 0 bits), use 2-color API:
  // This version paints 1-bits WHITE, 0-bits BLACK.
  display.drawBitmap(0, 0, epd_bitmap_Untitled_2, 128, 64, WHITE, BLACK);
}

void drawBall(int16_t bx, int16_t by) {
  display.drawBitmap(bx, by, epd_bitmap_8888, BALL_W, BALL_H, WHITE);
}

// ------- Physics -------
void ensureTopReachSpeed(float angleRad) {
  // ensure vy0 large enough so apex reaches y<=0 visually
  float y0 = y; // current top-left y of ball
  float needVy = sqrtf(2.0f * GRAVITY * y0) + 0.4f; // margin
  float sMin = needVy / sinf(angleRad);
  float base = 7.2f; // nominal
  float s = (sMin > base) ? (sMin) : base;

  vx = s * cosf(angleRad);
  vy = -s * sinf(angleRad);
}

void launchBallRandom() {
  // Random upwards angle (70..110 deg) -> more vertical so sure top hit
  float deg = random(70, 111);
  float rad = deg * 3.14159f / 180.0f;
  ensureTopReachSpeed(rad);
  onGround = false;
}

void physicsStep() {
  // gravity
  vy += GRAVITY;

  // integrate
  x += vx;
  y += vy;

  // side walls
  if (x <= 0) {
    x = 0; vx = -vx * DAMP_WALL;
  }
  if (x + BALL_W >= SCREEN_WIDTH) {
    x = SCREEN_WIDTH - BALL_W; vx = -vx * DAMP_WALL;
  }

  // top
  if (y <= 0) {
    y = 0; vy = -vy * DAMP_TOP;
  }

  // ground (no line drawn; rest when touching bottom row)
  float groundTop = (GROUND_Y - BALL_H); // 63 - 16 = 47
  if (y >= groundTop) {
    y = groundTop;
    vx *= 0.90f;
    vy = 0;
    onGround = true;
  }
}

void setup() {
  pinMode(BUTTON_PIN, INPUT_PULLUP);
  Wire.begin(D2, D1);

  if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) {
    if (!display.begin(SSD1306_SWITCHCAPVCC, 0x3D)) {
      for(;;);
    }
  }
  display.clearDisplay();
  display.display();

  randomSeed(analogRead(A0));

  // Welcome screen (just title), then start game
  drawWelcome();
  // drawWelcome2();
  delay(1200); // adjust if needed

  // init ball on base center
  x = (SCREEN_WIDTH - BALL_W)/2.0f;
  y = (GROUND_Y - BALL_H);
}

void loop() {
  // input: launch only from ground
  if (onGround && buttonJustPressed()) {
    launchBallRandom();
  }

  if (!onGround) {
    physicsStep();
  }

  // frame render: only background + ball
  display.clearDisplay();
  drawBackground();
  drawBall((int16_t)x, (int16_t)y);
  display.display();

  delay(16);
}
